<?php
// Arquivo: Comando_control.php
$configFile = __DIR__ . '/api/config.json';
$message = '';

// Carrega configurações existentes
$currentConfig = [
    'polling_enabled' => true,
    'poll_interval_ms' => 14400000 // 4 horas padrão
];

if (file_exists($configFile)) {
    $currentConfig = json_decode(file_get_contents($configFile), true) ?: $currentConfig;
}

// Processa o formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pollingEnabled = isset($_POST['polling_enabled']);
    $hours = intval($_POST['hours'] ?? 4);
    
    // Validação
    if ($hours < 1) $hours = 1;
    if ($hours > 24) $hours = 24;
    
    $newConfig = [
        'polling_enabled' => $pollingEnabled,
        'poll_interval_ms' => $hours * 3600000
    ];
    
    if (file_put_contents($configFile, json_encode($newConfig, JSON_PRETTY_PRINT))) {
        $currentConfig = $newConfig;
        $message = "<i class='fa fa-check-circle'></i> Configurações atualizadas com sucesso!";
    } else {
        $message = "<i class='fa fa-exclamation-circle'></i> Erro ao salvar configurações.";
    }
}
?>
<?php include './includes/header.php'; ?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Controle de Comando</title>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome/css/font-awesome.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #4CAF50;
            --primary-hover: #45a049;
            --toggle-on: #4CAF50;
            --toggle-off: #ccc;
            --toggle-knob: #fff;
            --warning-color: #ff9800;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
        }
        
        .container {
            background-color: #ffffff;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
            border-radius: 12px;
            padding: 2.5rem;
            width: 100%;
            max-width: 500px;
        }
        
        h1 {
            color: #2c3e50;
            text-align: center;
            margin-bottom: 1.8rem;
            font-size: 1.8rem;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .form-group {
            margin-bottom: 1.8rem;
        }
        
        label {
            color: #34495e;
            margin-bottom: 0.6rem;
            display: block;
            font-weight: 600;
            font-size: 1rem;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .toggle-container {
            display: flex;
            align-items: center;
            margin-bottom: 2rem;
            gap: 15px;
        }
        
        .toggle-label {
            font-weight: 600;
            color: #34495e;
            width: 140px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 60px;
            height: 34px;
        }
        
        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        
        .toggle-slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: var(--toggle-off);
            transition: .4s;
            border-radius: 34px;
        }
        
        .toggle-slider:before {
            position: absolute;
            content: "";
            height: 26px;
            width: 26px;
            left: 4px;
            bottom: 4px;
            background-color: var(--toggle-knob);
            transition: .4s;
            border-radius: 50%;
        }
        
        input:checked + .toggle-slider {
            background-color: var(--toggle-on);
        }
        
        input:checked + .toggle-slider:before {
            transform: translateX(26px);
        }
        
        .toggle-status {
            margin-left: auto;
            font-weight: bold;
            min-width: 100px;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .toggle-status.on {
            color: var(--toggle-on);
        }
        
        .toggle-status.off {
            color: #f44336;
        }
        
        input[type="number"] {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.3s;
            box-sizing: border-box;
            font-weight: 500;
        }
        
        input[type="number"]:focus {
            border-color: var(--primary-color);
            outline: none;
            box-shadow: 0 0 0 2px rgba(76, 175, 80, 0.2);
        }
        
        .input-container {
            position: relative;
            display: flex;
            align-items: center;
        }
        
        .input-container i {
            position: absolute;
            right: 15px;
            color: #7f8c8d;
        }
        
        button {
            background-color: var(--primary-color);
            color: white;
            border: none;
            padding: 14px 20px;
            font-size: 1rem;
            border-radius: 8px;
            cursor: pointer;
            width: 100%;
            transition: all 0.3s;
            font-weight: 600;
            margin-top: 1.5rem;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        button:hover {
            background-color: var(--primary-hover);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        button:active {
            transform: translateY(0);
        }
        
        .message {
            text-align: center;
            padding: 14px;
            margin-bottom: 2rem;
            border-radius: 8px;
            font-size: 0.95rem;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .success {
            background-color: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c8e6c9;
        }
        
        .error {
            background-color: #ffebee;
            color: #c62828;
            border: 1px solid #ffcdd2;
        }
        
        .status {
            margin-top: 2.5rem;
            padding: 1.5rem;
            border-radius: 10px;
            background-color: #f8f9fa;
            border: 1px solid #eee;
        }
        
        .status h3 {
            margin-top: 0;
            color: #2c3e50;
            font-size: 1.3rem;
            margin-bottom: 1.2rem;
            text-align: center;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .status-item {
            margin: 0.8rem 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .status-label {
            font-weight: 600;
            color: #34495e;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .status-value {
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .status-on {
            color: var(--toggle-on);
            font-weight: 600;
        }
        
        .status-off {
            color: #f44336;
            font-weight: 600;
        }
        
        .info-text {
            font-size: 0.85rem;
            color: #7f8c8d;
            margin-top: 6px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        /* Responsividade */
        @media (max-width: 600px) {
            .container {
                padding: 1.8rem;
                margin: 1rem;
            }
            
            h1 {
                font-size: 1.5rem;
            }
            
            .toggle-container {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
            
            .toggle-label {
                margin-bottom: 0.5rem;
                margin-right: 0;
            }
            
            .toggle-status {
                margin-left: 0;
            }
            
            .status-item {
                flex-direction: column;
                align-items: flex-start;
                gap: 4px;
            }
            
            button {
                padding: 12px 16px;
            }
        }
    </style>
</head>
<body>

<div class="container">
    <h1><i class="fa fa-sync-alt fa-spin" style="--fa-animation-duration: 3s;"></i> Controle de Comando</h1>

    <?php if ($message): ?>
        <div class="message <?= strpos($message, 'sucesso') !== false ? 'success' : 'error' ?>">
            <?= $message ?>
        </div>
    <?php endif; ?>

    <form method="post">
        <div class="toggle-container">
            <span class="toggle-label"><i class="fa fa-power-off"></i> Status do Comando:</span>
            <label class="toggle-switch">
                <input type="checkbox" name="polling_enabled" <?= $currentConfig['polling_enabled'] ? 'checked' : '' ?>>
                <span class="toggle-slider"></span>
            </label>
            <span class="toggle-status <?= $currentConfig['polling_enabled'] ? 'on' : 'off' ?>">
                <i class="fa <?= $currentConfig['polling_enabled'] ? 'fa-check-circle' : 'fa-times-circle' ?>"></i>
                <?= $currentConfig['polling_enabled'] ? 'ATIVADO' : 'DESATIVADO' ?>
            </span>
        </div>

        <div class="form-group">
            <label for="hours"><i class="fa fa-clock"></i> Intervalo de Tempo:</label>
            <div class="input-container">
                <input type="number" id="hours" name="hours" min="1" max="24" 
                       value="<?= $currentConfig['poll_interval_ms'] / 3600000 ?>" required>
                <i class="fa fa-hourglass-half"></i>
            </div>
            <div class="info-text">
                <i class="fa fa-info-circle"></i> Digite um valor entre 1 e 24 horas
            </div>
        </div>

        <button type="submit"><i class="fa fa-save"></i> SALVAR CONFIGURAÇÕES</button>
    </form>

    <div class="status">
        <h3><i class="fa fa-chart-bar"></i> Status Atual</h3>
        
        <div class="status-item">
            <span class="status-label"><i class="fa fa-power-off"></i> Comando:</span>
            <span class="status-value <?= $currentConfig['polling_enabled'] ? 'status-on' : 'status-off' ?>">
                <i class="fa <?= $currentConfig['polling_enabled'] ? 'fa-check' : 'fa-times' ?>"></i>
                <?= $currentConfig['polling_enabled'] ? 'Ativado' : 'Desativado' ?>
            </span>
        </div>
        
        <div class="status-item">
            <span class="status-label"><i class="fa fa-clock"></i> Intervalo:</span>
            <span class="status-value">
                <i class="fa fa-hourglass"></i>
                <?= $currentConfig['poll_interval_ms'] / 3600000 ?> horas
            </span>
        </div>
        
        <div class="status-item">
            <span class="status-label"><i class="fa fa-calendar-check"></i> Próxima verificação:</span>
            <span class="status-value">
                <i class="fa fa-history"></i>
                ~<?= $currentConfig['poll_interval_ms'] / 3600000 ?> horas
            </span>
        </div>
    </div>
</div>

<script>
    // Atualiza o texto e ícone do status quando o toggle é alterado
    document.querySelector('.toggle-switch input').addEventListener('change', function() {
        const statusElement = document.querySelector('.toggle-status');
        const icon = statusElement.querySelector('i');
        
        if (this.checked) {
            statusElement.textContent = 'ATIVADO';
            statusElement.className = 'toggle-status on';
            icon.className = 'fa fa-check-circle';
            statusElement.insertAdjacentElement('afterbegin', icon);
        } else {
            statusElement.textContent = 'DESATIVADO';
            statusElement.className = 'toggle-status off';
            icon.className = 'fa fa-times-circle';
            statusElement.insertAdjacentElement('afterbegin', icon);
        }
    });
</script>

</body>
</html>

<?php include './includes/footer.php'; ?>